#!/bin/bash

# =====================================================
# Script de Deploy - Cotizador AFL Cargo con Backend
# Dominio: aflcargo.net
# =====================================================

set -e

echo "🚀 Iniciando deploy del Cotizador AFL Cargo..."
echo "=============================================="

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}❌ Este script debe ejecutarse como root${NC}"
    echo "Ejecuta: sudo bash deploy.sh"
    exit 1
fi

DOMAIN="aflcargo.net"
WEB_ROOT="/var/www/$DOMAIN"
NGINX_CONF="/etc/nginx/sites-available/$DOMAIN"

echo -e "${YELLOW}📦 Paso 1: Actualizando sistema...${NC}"
apt update -qq

echo -e "${YELLOW}📦 Paso 2: Instalando Nginx y PHP...${NC}"
apt install nginx php-fpm php-mysql php-mbstring -y -qq
systemctl enable nginx
systemctl enable php8.1-fpm 2>/dev/null || systemctl enable php-fpm

echo -e "${YELLOW}📁 Paso 3: Creando directorios...${NC}"
mkdir -p $WEB_ROOT/cotizar
mkdir -p $WEB_ROOT/widget
mkdir -p $WEB_ROOT/api
mkdir -p $WEB_ROOT/admin

echo -e "${YELLOW}📄 Paso 4: Copiando archivos...${NC}"

if [ ! -f "cotizar.html" ]; then
    echo -e "${RED}❌ No se encontró cotizar.html${NC}"
    exit 1
fi

cp cotizar.html $WEB_ROOT/cotizar/index.html
cp widget/afl-quote-widget.js $WEB_ROOT/widget/ 2>/dev/null || echo "Widget no encontrado"
cp api/cotizacion-publica.php $WEB_ROOT/api/ 2>/dev/null || echo "API no encontrada"
cp admin/solicitudes.php $WEB_ROOT/admin/ 2>/dev/null || echo "Admin no encontrado"

echo -e "${YELLOW}🔧 Paso 5: Configurando Nginx con PHP...${NC}"

# Detectar versión de PHP-FPM
PHP_SOCK=$(ls /var/run/php/php*-fpm.sock 2>/dev/null | head -1)
if [ -z "$PHP_SOCK" ]; then
    PHP_SOCK="/var/run/php/php-fpm.sock"
fi

cat > $NGINX_CONF << NGINX_CONFIG
server {
    listen 80;
    listen [::]:80;
    
    server_name $DOMAIN www.$DOMAIN;
    root $WEB_ROOT;
    index index.php index.html;

    # Cotizador principal
    location /cotizar {
        alias $WEB_ROOT/cotizar;
        try_files \$uri \$uri/ /cotizar/index.html;
    }

    # Widget embebible
    location /widget {
        alias $WEB_ROOT/widget;
        add_header Access-Control-Allow-Origin *;
        add_header Access-Control-Allow-Methods 'GET, OPTIONS';
        expires 1d;
    }

    # API PHP
    location /api {
        alias $WEB_ROOT/api;
        location ~ \.php\$ {
            include snippets/fastcgi-php.conf;
            fastcgi_param SCRIPT_FILENAME \$request_filename;
            fastcgi_pass unix:$PHP_SOCK;
        }
    }

    # Admin PHP
    location /admin {
        alias $WEB_ROOT/admin;
        location ~ \.php\$ {
            include snippets/fastcgi-php.conf;
            fastcgi_param SCRIPT_FILENAME \$request_filename;
            fastcgi_pass unix:$PHP_SOCK;
        }
    }

    # Página principal
    location = / {
        return 301 /cotizar;
    }

    access_log /var/log/nginx/$DOMAIN.access.log;
    error_log /var/log/nginx/$DOMAIN.error.log;

    gzip on;
    gzip_types text/plain text/css application/json application/javascript;

    location ~ /\. {
        deny all;
    }
}
NGINX_CONFIG

ln -sf $NGINX_CONF /etc/nginx/sites-enabled/
nginx -t

echo -e "${YELLOW}🔄 Paso 6: Reiniciando servicios...${NC}"
systemctl reload nginx
systemctl restart php8.1-fpm 2>/dev/null || systemctl restart php-fpm

chown -R www-data:www-data $WEB_ROOT
chmod -R 755 $WEB_ROOT

echo -e "${GREEN}=============================================="
echo "✅ Deploy completado!"
echo "=============================================="
echo ""
echo "📍 URLs disponibles:"
echo "   Cotizador: http://$DOMAIN/cotizar"
echo "   Admin:     http://$DOMAIN/admin/solicitudes.php"
echo "   API:       http://$DOMAIN/api/cotizacion-publica.php"
echo ""
echo "⚠️  IMPORTANTE - Configurar base de datos:"
echo "   1. Importar database/cotizaciones_publicas.sql"
echo "   2. Editar credenciales en:"
echo "      - /var/www/$DOMAIN/api/cotizacion-publica.php"
echo "      - /var/www/$DOMAIN/admin/solicitudes.php"
echo ""
echo "🔒 Para HTTPS ejecuta:"
echo "   certbot --nginx -d $DOMAIN -d www.$DOMAIN"
echo ""
echo "📱 Cambiar WhatsApp:"
echo "   nano $WEB_ROOT/cotizar/index.html"
echo "===============================================${NC}"
